document.addEventListener("DOMContentLoaded", async function () {
  setUpSelectArea();
});

/* ------------------- page selection area code  -------------- */

function setUpSelectArea() {
  const selectAreaBtn = document.getElementById("selectAreaBtn");

  let popupSelecting = false;

  // Handle select area button click
  selectAreaBtn.addEventListener("click", function () {
    console.log("Select Area button clicked");
    if (popupSelecting) {
      console.log("Already selecting, ignoring click");
      return;
    }

    console.log("Starting area selection...");

    selectAreaBtn.textContent = "Selecting...";
    selectAreaBtn.disabled = true;

    // Start area selection directly in the active tab
    startAreaSelection();
    
    // Close the popup window
    window.close();
  });

  function startAreaSelection() {
    console.log("startAreaSelection called");
    if (popupSelecting) {
      console.log("Already selecting, returning");
      return;
    }

    popupSelecting = true;

    console.log("Querying active tabs...");
    // Inject the selection functionality into the active tab
    chrome.tabs.query({ active: true, currentWindow: true }, function (tabs) {
      console.log("Active tabs found:", tabs);
      if (!tabs[0] || !tabs[0].id) {
        console.error("No active tab found");
        resetButton();
        return;
      }

      console.log("Injecting into tab:", tabs[0].id);

      // Execute the code directly in the page context
      chrome.scripting
        .executeScript({
          target: { tabId: tabs[0].id },
          func: function () {
            console.log("Starting area selection injection...");

            // Test if we can access the DOM
            if (!document || !document.body) {
              alert("ERROR: Cannot access document or document.body!");
              return;
            }

            // Create the selection functionality directly
            let pageIsSelecting = false;
            let startX, startY;
            let selectionBox = null;
            let overlay = null;
            let isSelectionComplete = false;
            let currentSelection = null;

            function startSelection() {
              console.log("startSelection called");
              if (pageIsSelecting) return;

              pageIsSelecting = true;

              // close extension popup
              //chrome.runtime.sendMessage({ action: "closePopup" });

              console.log("Creating overlay...");

              // Create overlay
              createOverlay();

              // Add event listeners
              document.addEventListener("mousedown", handleMouseDown);
              document.addEventListener("mousemove", handleMouseMove);
              document.addEventListener("mouseup", handleMouseUp);
              document.addEventListener("keydown", handleKeyDown);

              // Change cursor
              document.body.style.cursor = "crosshair";

              // Prevent text selection
              document.addEventListener("selectstart", preventSelection);

              // Add visual feedback
              // Create a container for feedback and confirm button
              const feedbackContainer = document.createElement("div");
              feedbackContainer.style.cssText =
                "position: fixed; top: 0px; right: 0px; z-index: 1000001; display: flex; gap: 8px; pointer-events: auto; background: rgba(255,255,255,0); padding: 10px; border-radius: 8px; box-shadow: 0 4px 20px rgba(0,0,0,0.3);";

              // const feedback = document.createElement("div");
              // feedback.style.cssText =
              //   "background-color:#f44242; color: white; padding: 8px 12px; border-radius: 6px; font-family: Arial, sans-serif; font-size: 12px; box-shadow: 0 2px 10px rgba(0,0,0,0.3);";
              // feedback.textContent = "🖱️ Drag to select area";
              // feedbackContainer.appendChild(feedback);
              // console.log("Feedback element added");

              // add a button for confirm and copy to clipboard
              const confirmButton = document.createElement("button");
              //confirmButton.textContent = "Confirm Selection";
              confirmButton.textContent = "🖱️ Drag to select";
              confirmButton.id = "confirmSelectionBtn";
              confirmButton.style.cssText =
                "background-color:#f44242; color: white; padding: 8px 12px; border-radius: 6px; font-family: Arial, sans-serif; font-size: 12px; box-shadow: 0 2px 10px rgba(0,0,0,0.3); cursor: pointer; transition: all 0.2s ease; outline:none; border:none;";

              // Add hover effect
              confirmButton.addEventListener("mouseenter", function () {
                this.style.transform = "scale(1.05)";
                this.style.boxShadow = "0 4px 15px rgba(0,0,0,0.4)";
              });

              confirmButton.addEventListener("mouseleave", function () {
                this.style.transform = "scale(1)";
                this.style.boxShadow = "0 2px 10px rgba(0,0,0,0.3)";
              });

              feedbackContainer.appendChild(confirmButton);

              document.body.appendChild(feedbackContainer);

              confirmButton.addEventListener("mousedown", function (e) {
                console.log("Confirm button mousedown event fired!");
                e.stopPropagation();
              });

              confirmButton.addEventListener("click", async function (e) {
                e.preventDefault();
                e.stopPropagation();

                if (currentSelection) {
                  console.log("Opening results window...");
                  const formatedJsonData = JSON.stringify(currentSelection);
                  const base64 = btoa(formatedJsonData);
                  cleanup();
                  // copy to clipboard
                  await navigator.clipboard.writeText(base64);
                  alert("Code copied to clipboard");

                  //openResultsWindow(currentSelection);
                } else {
                  console.log("No current selection, showing alert");
                  alert("Please select an area first!");
                }
              });

              // Initially disable confirm button until selection is made
              confirmButton.disabled = true;

              // Store reference for cleanup
              overlay.feedbackContainer = feedbackContainer;
            }

            function createOverlay() {
              console.log("createOverlay called");
              // Create full-screen overlay
              overlay = document.createElement("div");
              overlay.style.cssText =
                "position: fixed; top: 0; left: 0; width: 100vw; height: 100vh; z-index: 999999; pointer-events: auto; background-color: rgba(0, 0, 0, 0.1);";
              document.body.appendChild(overlay);
              console.log("Overlay added to body");

              // Create selection box
              selectionBox = document.createElement("div");
              selectionBox.style.cssText =
                "position: fixed; border: 2px solid #4285f4; background-color: rgba(66, 133, 244, 0.2); pointer-events: none; z-index: 1000000; display: none; box-shadow: 0 0 10px rgba(66, 133, 244, 0.5);";
              overlay.appendChild(selectionBox);
              console.log("Selection box created");
            }

            function handleMouseDown(e) {
              console.log("Mouse down event");
              if (!pageIsSelecting) return;

              // If there's already a selection, clear it for new selection
              if (isSelectionComplete) {
                clearSelection();
              }

              startX = e.clientX;
              startY = e.clientY;
              console.log("Start position:", startX, startY);

              selectionBox.style.display = "block";
              selectionBox.style.left = startX + "px";
              selectionBox.style.top = startY + "px";
              selectionBox.style.width = "0px";
              selectionBox.style.height = "0px";
            }

            function handleMouseMove(e) {
              if (!pageIsSelecting || !startX) return;

              const currentX = e.clientX;
              const currentY = e.clientY;

              const left = Math.min(startX, currentX);
              const top = Math.min(startY, currentY);
              const width = Math.abs(currentX - startX);
              const height = Math.abs(currentY - startY);

              selectionBox.style.left = left + "px";
              selectionBox.style.top = top + "px";
              selectionBox.style.width = width + "px";
              selectionBox.style.height = height + "px";
            }

            function handleMouseUp(e) {
              console.log("Mouse up event");
              if (!pageIsSelecting || !startX) return;

              const endX = e.clientX;
              const endY = e.clientY;

              // Calculate dimensions
              const width = Math.abs(endX - startX);
              const height = Math.abs(endY - startY);
              const x = Math.min(startX, endX);
              const y = Math.min(startY, endY);
              const area = width * height;

              console.log("Selection completed:", {
                width,
                height,
                x,
                y,
                area,
              });

              // Store the current selection
              currentSelection = {
                width: width,
                height: height,
                x: x,
                y: y,
                windowWidth: window.innerWidth,
                windowHeight: window.innerHeight,
              };

              // Mark selection as complete
              isSelectionComplete = true;

              // Update feedback text
              // const feedback = overlay.feedbackContainer.querySelector("div");
              // if (feedback) {
              //   feedback.textContent = `✅ Area selected`;
              // }

              // Enable confirm button - use ID for reliable selection
              const confirmButton = overlay.feedbackContainer.querySelector(
                "#confirmSelectionBtn"
              );
              // console.log("Looking for confirm button with ID confirmSelectionBtn");
              // console.log("Feedback container:", overlay.feedbackContainer);
              // console.log("All buttons in container:", overlay.feedbackContainer.querySelectorAll("button"));

              if (confirmButton) {
                confirmButton.disabled = false;
                confirmButton.textContent = "Confirm Selection";
                confirmButton.style.backgroundColor = "#01b161";
                // console.log("Confirm button enabled successfully");
                // console.log("Button element:", confirmButton);
                // console.log("Button disabled state after enabling:", confirmButton.disabled);
              } else {
                console.error("Could not find confirm button to enable");
                console.log(
                  "Available buttons:",
                  Array.from(
                    overlay.feedbackContainer.querySelectorAll("button")
                  ).map((b) => ({ id: b.id, text: b.textContent }))
                );
              }

              // Keep selection visible - don't cleanup yet
              // Reset start coordinates for potential new selection
              startX = null;
              startY = null;
            }

            function clearSelection() {
              console.log("clearSelection called");
              isSelectionComplete = false;
              currentSelection = null;
              startX = null;
              startY = null;

              // Hide selection box
              if (selectionBox) {
                selectionBox.style.display = "none";
              }

              // Reset feedback text
              // const feedback = overlay.feedbackContainer.querySelector("div");
              // if (feedback) {
              //   feedback.textContent = "🖱️ Drag to select area";
              // }

              // Disable confirm button - use ID for reliable selection
              const confirmButton = overlay.feedbackContainer.querySelector(
                "#confirmSelectionBtn"
              );
              if (confirmButton) {
                confirmButton.disabled = true;
                console.log("Confirm button disabled");
              } else {
                console.error("Could not find confirm button to disable");
              }
            }

            function handleKeyDown(e) {
              if (pageIsSelecting && e.key === "Escape") {
                console.log("Escape key pressed, cleaning up");
                cleanup();
              }
            }

            function cleanup() {
              console.log("cleanup called");
              pageIsSelecting = false;
              isSelectionComplete = false;
              startX = null;
              startY = null;
              currentSelection = null;

              // Remove event listeners
              document.removeEventListener("mousedown", handleMouseDown);
              document.removeEventListener("mousemove", handleMouseMove);
              document.removeEventListener("mouseup", handleMouseUp);
              document.removeEventListener("keydown", handleKeyDown);
              document.removeEventListener("selectstart", preventSelection);

              // Remove overlay and selection box
              if (overlay) {
                if (overlay.feedbackContainer) {
                  overlay.feedbackContainer.remove();
                }
                overlay.remove();
                overlay = null;
              }
              if (selectionBox) {
                selectionBox = null;
              }

              // Reset cursor
              document.body.style.cursor = "";
            }

            function preventSelection(e) {
              if (pageIsSelecting) {
                e.preventDefault();
                return false;
              }
            }

            // Start the selection process immediately
            console.log("Starting selection process...");
            startSelection();
            console.log("Selection process started");

            // Send a message to the popup when selection is complete
            window.addEventListener("message", function (event) {
              if (event.data.type === "selectionComplete") {
                console.log("Selection completed, notifying popup");
                // This will be handled by the popup
              }
            });
          },
        })
        .then(() => {
          console.log("Area selection injected successfully");
          // Reset the popup button state after successful injection
          resetButton();
        })
        .catch((error) => {
          console.error("Failed to inject area selection:", error);
          resetButton();
        });
    });
  }

  function resetButton() {
    popupSelecting = false;
    selectAreaBtn.textContent = "Select Area";
    selectAreaBtn.disabled = false;
  }

  // Check if we're on a valid page
  chrome.tabs.query({ active: true, currentWindow: true }, function (tabs) {
    if (
      tabs[0].url.startsWith("chrome://") ||
      tabs[0].url.startsWith("chrome-extension://")
    ) {
      selectAreaBtn.disabled = true;
      selectAreaBtn.textContent = "Not available on this page";
    }
  });
}
